from apps.app import db
from datetime import datetime

class TrackingEvent(db.Model):
    __tablename__ = 'tracking_events'

    id = db.Column(db.Integer, primary_key=True)
    
    # Visitor/session data
    visitor_id = db.Column(db.String(64), nullable=True)  # Could be from cookie or fingerprint
    ip_address = db.Column(db.String(45), nullable=True)
    user_agent = db.Column(db.Text, nullable=True)
    browser = db.Column(db.String(120), nullable=True)
    device = db.Column(db.String(120), nullable=True)
    os = db.Column(db.String(120), nullable=True)
    language = db.Column(db.String(32), nullable=True)

    # Location data
    country = db.Column(db.String(64), nullable=True)
    region = db.Column(db.String(64), nullable=True)
    city = db.Column(db.String(64), nullable=True)
    latitude = db.Column(db.Float, nullable=True)
    longitude = db.Column(db.Float, nullable=True)

    # Page and interaction data
    page_url = db.Column(db.Text, nullable=False)
    page_title = db.Column(db.String(255), nullable=True)
    referrer = db.Column(db.Text, nullable=True)
    num_clicks = db.Column(db.Integer, default=0)
    scroll_depth = db.Column(db.Float, default=0.0)  # Percentage (0.0 to 1.0)
    time_spent = db.Column(db.Float, default=0.0)  # In seconds

    # Misc
    event_type = db.Column(db.String(64), nullable=True)  # e.g., 'visit', 'click', 'scroll'
    custom_data = db.Column(db.JSON, nullable=True)  # Additional optional metadata

    # Timestamps
    timestamp = db.Column(db.DateTime, default=datetime.utcnow)


    @staticmethod
    def addEvent(session, data, request):
        try:
            event = TrackingEvent(
                session_id=session.get('tracking_id'),
                user_id=None,  # Optional: Add logic if auth exists
                event_type=data.get("event_type"),
                page_path=data.get("page_path"),
                referrer=data.get("referrer", request.referrer),
                user_agent=request.headers.get("User-Agent"),
                ip_address=request.remote_addr,
                object_type=data.get("object_type"),
                object_id=data.get("object_id"),
                event_metadata =data.get("metadata", {}),
                timestamp=datetime.now()
            )
            db.session.add(event)
            db.session.commit()
            
            return True, 'Event added successfully!'
        
        except Exception as e:
            return False,  str(e)