// zentrack.js - Inject this into client websites
(function () {
  const ZEN_ENDPOINT = 'https://your-zenalytics.com/collect';

  function getFingerprint() {
    return btoa([
      navigator.userAgent,
      screen.width,
      screen.height,
      screen.colorDepth,
      navigator.language,
      new Date().getTimezoneOffset()
    ].join(':'));
  }

  function sendEvent(data) {
    navigator.sendBeacon(ZEN_ENDPOINT, JSON.stringify(data));
  }

  const sessionId = sessionStorage.getItem('zen_sid') || crypto.randomUUID();
  sessionStorage.setItem('zen_sid', sessionId);

  const baseData = {
    type: 'pageview',
    fingerprint: getFingerprint(),
    userAgent: navigator.userAgent,
    language: navigator.language,
    referrer: document.referrer,
    url: window.location.href,
    title: document.title,
    screen: {
      width: screen.width,
      height: screen.height,
      colorDepth: screen.colorDepth
    },
    sessionId,
    timestamp: new Date().toISOString(),
    utm: Object.fromEntries(new URLSearchParams(location.search).entries())
  };

  sendEvent(baseData);

  // Scroll depth
  let maxScroll = 0;
  window.addEventListener('scroll', () => {
    const scrollTop = window.scrollY;
    const docHeight = document.documentElement.scrollHeight - window.innerHeight;
    const scrolled = (scrollTop / docHeight) * 100;
    maxScroll = Math.max(scrolled, maxScroll);
  });

  window.addEventListener('beforeunload', () => {
    sendEvent({
      type: 'engagement',
      event: 'scroll-depth',
      scroll: maxScroll,
      timestamp: new Date().toISOString(),
      sessionId,
      fingerprint: getFingerprint(),
      url: window.location.href
    });
  });

  // Click events
  document.addEventListener('click', e => {
    const el = e.target.closest('[data-track-click]');
    if (el) {
      sendEvent({
        type: 'click',
        element: el.tagName,
        text: el.innerText.slice(0, 100),
        id: el.id,
        class: el.className,
        sessionId,
        timestamp: new Date().toISOString(),
        url: window.location.href,
        fingerprint: getFingerprint()
      });
    }
  });

  // JS errors
  window.addEventListener('error', (e) => {
    sendEvent({
      type: 'error',
      message: e.message,
      filename: e.filename,
      lineno: e.lineno,
      colno: e.colno,
      timestamp: new Date().toISOString(),
      sessionId,
      fingerprint: getFingerprint(),
      url: window.location.href
    });
  });
})();
