// analytics.js
(function () {
    const endpoint = 'https://yourdomain.com/collect'; // update with actual endpoint

    function uuidv4() {
        return ([1e7]+-1e3+-4e3+-8e3+-1e11).replace(/[018]/g, c =>
            (c ^ crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16)
        );
    }

    function getSessionId() {
        let sessionId = sessionStorage.getItem('zenalytics_sid');
        if (!sessionId) {
            sessionId = uuidv4();
            sessionStorage.setItem('zenalytics_sid', sessionId);
        }
        return sessionId;
    }

    function getDeviceInfo() {
        return {
            userAgent: navigator.userAgent,
            platform: navigator.platform,
            language: navigator.language,
            screenWidth: screen.width,
            screenHeight: screen.height,
            timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
        };
    }

    function getPageInfo() {
        return {
            path: window.location.pathname,
            fullUrl: window.location.href,
            title: document.title,
            referrer: document.referrer
        };
    }

    function sendAnalyticsEvent(eventName, additionalData = {}) {
        const payload = {
            event: eventName,
            session_id: getSessionId(),
            timestamp: new Date().toISOString(),
            ...getDeviceInfo(),
            ...getPageInfo(),
            ...additionalData
        };

        navigator.sendBeacon(endpoint, JSON.stringify(payload));
    }

    function setupScrollTracker() {
        let maxScroll = 0;
        window.addEventListener('scroll', () => {
            const scrollDepth = Math.floor(window.scrollY / document.body.scrollHeight * 100);
            if (scrollDepth > maxScroll) maxScroll = scrollDepth;
        });
        window.addEventListener('beforeunload', () => {
            sendAnalyticsEvent('scroll_depth', { scroll_depth_percent: maxScroll });
        });
    }

    function setupClickTracker() {
        let clickCount = 0;
        document.addEventListener('click', () => {
            clickCount++;
        });
        window.addEventListener('beforeunload', () => {
            sendAnalyticsEvent('click_count', { total_clicks: clickCount });
        });
    }

    function setupTimeOnPageTracker() {
        const startTime = Date.now();
        window.addEventListener('beforeunload', () => {
            const duration = Math.floor((Date.now() - startTime) / 1000);
            sendAnalyticsEvent('time_on_page', { duration_seconds: duration });
        });
    }

    function setupErrorTracking() {
        window.addEventListener('error', (e) => {
            sendAnalyticsEvent('js_error', {
                message: e.message,
                source: e.filename,
                lineno: e.lineno,
                colno: e.colno
            });
        });

        window.addEventListener('unhandledrejection', (e) => {
            sendAnalyticsEvent('unhandled_promise_rejection', {
                reason: e.reason?.toString() || 'unknown error'
            });
        });
    }

    function initZenalytics() {
        sendAnalyticsEvent('pageview');
        setupScrollTracker();
        setupClickTracker();
        setupTimeOnPageTracker();
        setupErrorTracking();
    }

    document.addEventListener('DOMContentLoaded', initZenalytics);
})();
